/*
    proj.c

	converts x,y to lat,lon
	or lat,lon to x,y


	To use a ____.bat file to run you can name the output file
	and either make a new file or append data to an existing one.

	albers s10_str  str10_13  /n
	albers s11_str  str10_13  /a

	This will create a new file 'tr10_13.raw' and put both
	's10_str.txt' and 's11_str.txt' into it.

*/
#include <stdio.h>
#include <math.h>
#include <string.h>

#define PI		3.1415927
#define PI4		0.7853981
#define DEG_RAD 0.01745329251				/* converts degrees to radians */
#define NUM_PROJ 10
#define SKIP 	while(fgetc(fpin)!=10)


struct proj
{
	char name[20];
	int year;
	double a,b;
	char remark[100];
}Ellip[NUM_PROJ]=
{
	{"GRS 80",1980,6378137.0,6356752.3,"Newly adopted"},
	{"WGS 72",1972,6378135.0,6356750.5,"NASA, DoD, oil Co."},
	{"Australian",1965,6378160.0,6356774.7,"Australia"},
	{"Krasovsky",1940,6378245.0,6356863.0,"Soviet Union"},
	{"International",1924,6378388.0,6356911.9,"Remainer of the World"},
	{"Hayford",1924,6378388.0,6356911.9,"Remainer of the World"},
	{"Clarke",1880,6378249.1,6356514.9,"Most of Africa; France"},
	{"Clarke",1866,6378206.4,6356583.8,"North America; Philippines"},
	{"Airy",1830,6377563.4,6356256.9,"Great Britain"},
	{"Everest",1830,6377276.3,6356075.4,"India;Burma;Pak.;Afgan.;Thailand;etc."}
};
int Proj;
double e,a,e2,n,C,rho0,q_max;
double ep2,c0,c2,c4,c6,m0;

int set_p(void);
double Lat0,Lat1,Lat2,Lon0,K0;
int set_alb(int p,double lat0,double lat1,double lat2,double lon0);
int axy_ll_e(double x,double y,double *lat,double *lon);
int all_xy_e(double *x,double *y,double lat,double lon);
int set_etm(int p,double lat0,double lon0,double k0);
int txy_ll_e(double x,double y,double *lat,double *lon);
int tll_xy_e(double *x,double *y,double lat,double lon);

/***********************************************************************
**
**
**
**
************************************************************************/

int axy_ll_e(double x,double y,double *lat,double *lon)

{
	double q,rho,theta;
	double ry=rho0-y,err,sign=1.0,tlat,sin_tlat;
	int time=0,maxtime=100;

	if(n<0.0)sign=-1.0;
	theta=atan2(x*sign,ry*sign);
	rho=sqrt(x*x+ry*ry);
	q=(C-rho*rho*n*n/(a*a))/n;

	if(q>2.0||q<-2.0)
	{
		return(-1);
	}
	tlat=asin(q/2.0);

	if(q<-q_max||q>q_max)
	{
		return(-1);
	}
	else if(q==q_max)
		*lat=90.0*DEG_RAD;
	else if(q==-q_max)
		*lat=-90.0*DEG_RAD;
	else do
	{
		sin_tlat=sin(tlat);
		*lat=tlat+(1.0-e2*sin_tlat*sin_tlat)*(1.0-e2*sin_tlat*sin_tlat)/
			(2.0*cos(tlat))*
			(q/(1.0-e2)-sin(tlat)/(1.0-e2*sin_tlat*sin_tlat)+1/(2.0*e)*
			log((1.0-e*sin_tlat)/(1.0+e*sin_tlat)));
		err=(*lat-tlat)*(*lat-tlat);
		tlat=*lat;
	}while(err>0.000000001&&time++<maxtime);

	*lon=Lon0+theta/n;
	*lon/=DEG_RAD;
	*lat/=DEG_RAD;

	return(1.0);
}




/********************************************************************
**
**
**
********************************************************************* */

int all_xy_e(double *x,double *y,double lat,double lon)

{
	double q,m,rho,theta;

	if(lat>90.0||lat<-90.0)
		return(-1);

	lat*=DEG_RAD;
	lon*=DEG_RAD;

	q=(1.0-e*e)*(sin(lat)/(1.0-e*e*sin(lat)*sin(lat))-1.0/(2.0*e)*
		(log((1.0-e*sin(lat))/(1.0+e*sin(lat)))));
	m=cos(lat)/sqrt(1.0-e*e*sin(lat)*sin(lat));
	rho=a*sqrt(C-n*q)/n;
	theta=n*(lon-Lon0);
	*x=rho*sin(theta);
	*y=rho0-rho*cos(theta);

	return(1);
}


/********************************************************************
**
**	input projection number 
**	plus coordinates in degrees.
**
********************************************************************* */

int set_alb(int p,double lat0,double lat1,double lat2,double lon0)

{
	double m1,m2,q0,q1,q2;

	lat0*=DEG_RAD;
	lat1*=DEG_RAD;
	lat2*=DEG_RAD;
	lon0*=DEG_RAD;
	Lat0=lat0;
	Lat1=lat1;
	Lat2=lat2;
	Lon0=lon0;

	a=Ellip[p].a;
	e2=1.0-Ellip[p].b*Ellip[p].b/(Ellip[p].a*Ellip[p].a);
	e=sqrt(e2);
	q0=(1.0-e*e)*(sin(lat0)/(1.0-e*e*sin(lat0)*sin(lat0))-1.0/(2.0*e)*
		(log((1.0-e*sin(lat0))/(1.0+e*sin(lat0)))));
	q1=(1.0-e*e)*(sin(lat1)/(1.0-e*e*sin(lat1)*sin(lat1))-1.0/(2.0*e)*
		(log((1.0-e*sin(lat1))/(1.0+e*sin(lat1)))));
	q2=(1.0-e*e)*(sin(lat2)/(1.0-e*e*sin(lat2)*sin(lat2))-1.0/(2.0*e)*
		(log((1.0-e*sin(lat2))/(1.0+e*sin(lat2)))));
	m1=cos(lat1)/sqrt(1.0-e*e*sin(lat1)*sin(lat1));
	m2=cos(lat2)/sqrt(1.0-e*e*sin(lat2)*sin(lat2));
	if(lat1==lat2)
		n=sin(lat1);
	else
		n=(m1*m1-m2*m2)/(q2-q1);

	C=m1*m1+n*q1;
	rho0=a*sqrt(C-n*q0)/n;
	q_max=1.0-((1.0-e2)/(2.0*e))*log((1.0-e)/(1.0+e));
}


/********************************************************************
**
**
**
********************************************************************* */

int set_p()

{
	int i,j,k,p;

	do
	{
		for(i=0;i<NUM_PROJ;i++)
			printf("%2d%15s %4d %9.1lf %9.1lf %s\n",
				i+1,Ellip[i].name,Ellip[i].year,Ellip[i].a,Ellip[i].b,
				Ellip[i].remark);
		printf("\nEnter number of choice\n");
		scanf("%d",&p);
	}while(p<1||p>NUM_PROJ);
	return(p-1);
}


/********************************************************************
**
**
**
********************************************************************* */

int txy_ll_e(double x,double y,double *lat,double *lon)

{
    double m,e1,mu,phi1,c1,t1,n1,r1,d;

      m=m0 + y/K0;
      e1=(1.-sqrt(1.-e2))/(1.+sqrt(1.-e2));
      mu=m/(a*(1.-e2/4.-3.*e2*e2/64.-5.*e2*e2*e2/256.));
      phi1=mu+(3.*e1/2.-27.*e1*e1*e1/32.)*sin(2.*mu) +
           (21.*e1*e1/16.-55.*e1*e1*e1*e1/32.)*sin(4.*mu) +
           (151.*e1*e1*e1/96.)*sin(6.*mu);
      c1=ep2*cos(phi1)*cos(phi1);
      t1=tan(phi1)*tan(phi1);
      n1=a/sqrt(1.-e2*sin(phi1)*sin(phi1));
      r1=a*(1-e2)/((1-e2*sin(phi1)*sin(phi1))*sqrt(1-e2*sin(phi1)*sin(phi1)));
      d=x/(n1*K0);
   
      *lat=phi1-(n1*tan(phi1)/r1)*
          (d*d/2.-(5.+3.*t1+10.*c1-4.*c1*c1-9.*ep2)*d*d*d*d/24.
          + (61.+90*t1+298*c1+45*t1*t1-252.*ep2-3.*c1*c1)*d*d*d*d*d*d/720.);
      *lon=Lon0+(d-(1.+2.*t1+c1)*d*d*d/6.
          +(5.-2.*c1+28.*t1-3.*c1*c1+8.*ep2+24.*t1*t1)*d*d*d*d*d/120.)
          /cos(phi1);

      *lon/=DEG_RAD;
      *lat/=DEG_RAD;


	return(1.0);
}




/********************************************************************
**
**
**
********************************************************************* */

int tll_xy_e(double *x,double *y,double lat,double lon)

{
    double n,tnphi,t,csphi,c,biga,m;

	if(lat>90.0||lat<-90.0)
		return(-1);

	lat*=DEG_RAD;
	lon*=DEG_RAD;

      n=a/sqrt(1.-e2*sin(lat)*sin(lat));
      tnphi=tan(lat);
      t=tnphi*tnphi;
      csphi=cos(lat);
      c=ep2*csphi*csphi;
      biga=csphi*(lon-Lon0);
      m=a*(c0*lat-c2*sin(2.*lat)+c4*sin(4.*lat)-c6*sin(6.*lat));

      *x=K0*n*(biga+(1.-t+c)*biga*biga*biga/6. +
          (5.-18.*t+t*t+72.*c-58.*ep2)*biga*biga*biga*biga*biga/120.);
      *y=K0*(m-m0+n*tnphi*(.5*biga*biga +
          (5.-t+9.*c+4.*c*c)*biga*biga*biga*biga/24. +
          (61.-58.*t+t*t+600.*c-330.*ep2)*biga*biga*biga*biga*biga*biga/720.));

	return(1);
}


/********************************************************************
**
**	input projection number 
**	plus coordinates in degrees.
**
********************************************************************* */

int set_etm(int p,double lat0,double lon0,double k0)

{

	lat0*=DEG_RAD;
	lon0*=DEG_RAD;
	Lat0=lat0;
	Lon0=lon0;
    K0=k0;

	a=Ellip[p].a;
	e2=1.0-Ellip[p].b*Ellip[p].b/(Ellip[p].a*Ellip[p].a);
	e=sqrt(e2);
    ep2=e2/(1.-e2);
    c0=1.-e*e/4.-3.*e*e*e*e/64.-5.*e*e*e*e*e*e/256.;
    c2=3.*e*e/8.+3.*e*e*e*e/32.+45.*e*e*e*e*e*e/1024.;
    c4=15.*e*e*e*e/256.+45.*e*e*e*e*e*e/1024.;
    c6=35.*e*e*e*e*e*e/3072.;
    m0=a*(c0*lat0-c2*sin(2.*lat0)+c4*sin(4.*lat0)-c6*sin(6.*lat0));

}







             

